<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/config/db.php';
require_once __DIR__ . '/includes/toyyibpay_helper.php';
if (file_exists(__DIR__ . '/includes/whatsapp_helper.php')) {
    require_once __DIR__ . '/includes/whatsapp_helper.php';
}

date_default_timezone_set('Asia/Kuala_Lumpur');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo "Method Not Allowed";
    exit;
}

// 🧩 Ambil input
$buyer_name  = trim($_POST['buyer_name'] ?? '');
$buyer_phone = trim($_POST['buyer_phone'] ?? '');
$buyer_email = trim($_POST['buyer_email'] ?? '');
$kategori    = $_POST['kategori'] ?? '';
$size        = $_POST['size'] ?? '';
$quantity    = max(1, (int)($_POST['quantity'] ?? 1));
$addon       = isset($_POST['addon_lengan']) && $_POST['addon_lengan'] == '1';

// 🛡️ SERVER VALIDATION
$errors = [];
if ($buyer_name === '') $errors[] = "Nama wajib diisi.";
if (!preg_match('/^(?:\+?601\d{7,9})$/', $buyer_phone)) $errors[] = "No. telefon tidak sah. Gunakan format 601XXXXXXXX.";
if (!filter_var($buyer_email, FILTER_VALIDATE_EMAIL)) $errors[] = "Alamat emel tidak sah.";
if (!in_array($kategori, ['adult','muslimah','pasti'])) $errors[] = "Kategori tidak sah.";
if ($size === '') $errors[] = "Sila pilih saiz.";
if ($quantity < 1 || $quantity > 10) $errors[] = "Kuantiti mesti antara 1 hingga 10.";

if (!empty($errors)) {
    echo "<div style='max-width:600px;margin:50px auto;font-family:Arial;text-align:left;padding:30px;border:1px solid #eee;border-radius:10px;'>
            <h2 style='color:#c00;'>⚠️ Ralat Input</h2><ul style='color:#333;'>";
    foreach ($errors as $e) echo "<li>$e</li>";
    echo "</ul><a href='javascript:history.back()' style='display:inline-block;margin-top:15px;padding:10px 20px;background:#007bff;color:#fff;border-radius:5px;text-decoration:none;'>⬅️ Kembali</a></div>";
    exit;
}

// 💰 Kira harga
$price_map = ['adult'=>40,'muslimah'=>50,'pasti'=>5];
$unit_price = $price_map[$kategori] ?? 0;
$total = $unit_price * $quantity;
if ($addon) $total += 5 * $quantity;

// 🧾 Cipta order ID unik
$order_id = 'SHIRT-' . strtoupper(bin2hex(random_bytes(4)));

// 🧷 Simpan ke DB
$meta = json_encode([
    'product' => 'shirt',
    'kategori' => $kategori,
    'size' => $size,
    'quantity' => $quantity,
    'unit_price' => $unit_price,
    'addon_lengan' => $addon
]);

try {
    $stmt = $pdo->prepare("
        INSERT INTO sales (order_id, buyer_name, buyer_phone, buyer_email, amount, payment_status, meta, created_at)
        VALUES (:id, :name, :phone, :email, :amount, 'PENDING', :meta, NOW())
    ");
    $stmt->execute([
        ':id' => $order_id,
        ':name' => $buyer_name,
        ':phone' => $buyer_phone,
        ':email' => $buyer_email,
        ':amount' => $total,
        ':meta' => $meta
    ]);
} catch (Throwable $e) {
    file_put_contents(__DIR__ . '/logs/order_process_log.txt',
        "[" . date('Y-m-d H:i:s') . "] ❌ DB Insert Error: " . $e->getMessage() . "\n", FILE_APPEND);
    die("<h2 style='color:red'>Ralat menyimpan ke pangkalan data</h2>");
}

// 🔑 Konfigurasi ToyyibPay
$set = $pdo->query("SELECT * FROM payment_settings WHERE gateway_name='toyyibpay' LIMIT 1")->fetch();
$api_key  = $set['api_key'] ?? '';
$category = $set['category_code'] ?? '';
$env      = $set['environment'] ?? 'production';

$baseUrl = (isset($_SERVER['REQUEST_SCHEME']) ? $_SERVER['REQUEST_SCHEME'] : 'https') . '://' . $_SERVER['HTTP_HOST'];
$returnUrl   = $baseUrl . '/pasti/payment/return.php';
$callbackUrl = $baseUrl . '/pasti/payment/callback_toyyib.php';

// 🧾 Cipta bil ToyyibPay
$bill = createToyyibBill(
    $api_key,
    $category,
    $order_id,
    $buyer_name,
    $buyer_email,
    $buyer_phone,
    $total,
    $returnUrl,
    $callbackUrl,
    $env
);

// 🧾 Log respon ToyyibPay
$logFile = __DIR__ . '/logs/order_process_log.txt';
file_put_contents($logFile, "[" . date('Y-m-d H:i:s') . "] CREATE BILL RESPONSE:\n" . print_r($bill, true) . "\n", FILE_APPEND);

// 💬 WhatsApp & Redirect
if (!empty($bill['bill_url'])) {

    // Simpan bill_code ke DB
    try {
        $upd = $pdo->prepare("UPDATE sales SET bill_code = :code WHERE order_id = :id");
        $upd->execute([':code' => $bill['bill_code'], ':id' => $order_id]);
    } catch (Throwable $e) {
        file_put_contents($logFile, "[" . date('Y-m-d H:i:s') . "] ⚠️ Update bill_code error: " . $e->getMessage() . "\n", FILE_APPEND);
    }

    // Hantar WhatsApp template “order”
    if (function_exists('sendWhatsAppTemplate')) {
        sendWhatsAppTemplate('order', [
            'nama' => $buyer_name,
            'phone' => $buyer_phone,
            'order_id' => $order_id,
            'amount' => number_format($total, 2),
            'link_bayaran' => $bill['bill_url'],
            'tarikh' => date('d/m/Y H:i')
        ]);
    } else {
        file_put_contents($logFile, "[" . date('Y-m-d H:i:s') . "] ⚠️ Fungsi sendWhatsAppTemplate() tidak wujud.\n", FILE_APPEND);
    }

    // Redirect ke halaman pembayaran
    header('Location: ' . $bill['bill_url']);
    exit;

} else {
    // ❌ Gagal cipta bil
    file_put_contents($logFile, "[" . date('Y-m-d H:i:s') . "] ❌ Gagal cipta bil ToyyibPay.\n", FILE_APPEND);
    echo "<div style='max-width:600px;margin:50px auto;font-family:Arial;text-align:center;padding:30px;border:1px solid #eee;border-radius:10px;'>
            <h2 style='color:#c00;'>❌ Gagal Cipta Bil</h2>
            <p>Sila semak semula konfigurasi ToyyibPay atau API Key tidak sah.</p>
            <pre style='background:#f8f8f8;padding:10px;border-radius:5px;margin-top:10px;text-align:left;'>".print_r($bill,true)."</pre>
            <a href='javascript:history.back()' style='display:inline-block;margin-top:15px;padding:10px 20px;background:#007bff;color:#fff;border-radius:5px;text-decoration:none;'>⬅️ Kembali</a>
          </div>";
    exit;
}
?>
