<?php
require_once __DIR__ . '/../config/app.php';

/**
 * Dapatkan URL asas ToyyibPay ikut environment.
 */
function getToyyibBase($env)
{
    return ($env === 'sandbox')
        ? 'https://dev.toyyibpay.com'
        : 'https://toyyibpay.com';
}

/**
 * Cipta bil ToyyibPay.
 * @param string $api_key
 * @param string $category_code
 * @param string $order_id
 * @param string $buyer_name
 * @param string $buyer_email
 * @param string $buyer_phone
 * @param float  $amount
 * @param string $return_url
 * @param string $callback_url
 * @param string $env
 */
function createToyyibBill($api_key, $category_code, $order_id, $buyer_name, $buyer_email, $buyer_phone, $amount, $return_url, $callback_url, $env = 'production')
{
    $base = getToyyibBase($env);
    $url = rtrim($base, '/') . '/index.php/api/createBill';

    $payload = [
        'userSecretKey'           => $api_key,
        'categoryCode'            => $category_code,
        'billName'                => "Order $order_id",
        'billDescription'         => "Payment for order $order_id",
        'billPriceSetting'        => 1,
        'billAmount'              => (int) round($amount * 100),
        'billReturnUrl'           => $return_url,
        'billCallbackUrl'         => $callback_url,
        'billExternalReferenceNo' => $order_id,
        'billTo'                  => $buyer_name,
        'billEmail'               => $buyer_email,
        'billPhone'               => $buyer_phone,
        'billPaymentChannel'      => '0',
        'billPayorInfo'           => 1,
    ];

    $logPath = __DIR__ . '/../payment/toyyibpay_log.txt';
    file_put_contents($logPath, "[" . date('Y-m-d H:i:s') . "] CREATE BILL REQUEST\nENV: $env\nURL: $url\nPAYLOAD:\n" . print_r($payload, true) . "\n", FILE_APPEND);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $res = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err = curl_error($ch);
    curl_close($ch);

    file_put_contents($logPath, "[" . date('Y-m-d H:i:s') . "] RESPONSE\nHTTP_CODE: $http_code\nCURL_ERR: $err\nRESPONSE_BODY:\n$res\n\n", FILE_APPEND);

    if ($err) {
        return ['error' => 'Curl error: ' . $err, 'http_code' => $http_code];
    }

    $json = json_decode($res, true);

    if (is_array($json) && isset($json[0]['BillCode'])) {
        $billcode = $json[0]['BillCode'];
        $billurl = rtrim($base, '/') . '/' . $billcode;
        return [
            'bill_code' => $billcode,
            'bill_url'  => $billurl,
            'raw'       => $json,
            'http_code' => $http_code
        ];
    }

    return [
        'error' => 'Unexpected response from ToyyibPay',
        'http_code' => $http_code,
        'raw' => $res
    ];
}

/**
 * Semak status bil ToyyibPay.
 */
function verifyToyyibBill($api_key, $billcode, $env = 'production')
{
    $base = getToyyibBase($env);
    $url = $base . '/index.php/api/getBillTransactions';
    $data = [
        'userSecretKey' => $api_key,
        'billCode'      => $billcode
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    file_put_contents(__DIR__ . '/../payment/toyyibpay_log.txt',
        "[" . date('Y-m-d H:i:s') . "] VERIFY BILL\nURL: $url\nHTTP:$http\nERR:$err\nRES:\n$res\n\n",
        FILE_APPEND
    );

    if ($err) return ['error' => 'Curl error: ' . $err];
    $json = json_decode($res, true);
    return $json;
}
