<?php
// includes/pdf_receipt.php
// Generates a PDF receipt using Dompdf. Assumes composer autoload available at ../vendor/autoload.php
if (session_status() === PHP_SESSION_NONE) session_start();
require_once __DIR__ . '/../config/db.php';

function generateReceiptPDF($order) {
    // $order should contain keys: order_id, buyer_name, buyer_phone, amount, meta (array), payment_date
    $vendor = __DIR__ . '/../vendor/autoload.php';
    if (!file_exists($vendor)) {
        // Dompdf not installed - return error
        return ['ok'=>false,'error'=>'Dompdf not installed (vendor/autoload.php missing)'];
    }
    require_once $vendor;
    // use Dompdf
    if (!class_exists('\Dompdf\Dompdf')) {
        return ['ok'=>false,'error'=>'Dompdf class not found'];
    }
    $meta = is_string($order['meta']) ? json_decode($order['meta'], true) : ($order['meta'] ?? []);

    $logo_path = __DIR__ . '/../assets/logo.png';
    $logo_src = file_exists($logo_path) ? 'file://' . realpath($logo_path) : '';

    $html = '<html><head><meta charset="utf-8"><style>
      body{font-family: DejaVu Sans, Arial, sans-serif; font-size:14px; color:#222}
      .header{display:flex;align-items:center;gap:10px}
      .logo{width:80px}
      .title{font-size:18px;font-weight:700}
      .muted{color:#666;font-size:12px}
      table{width:100%;border-collapse:collapse;margin-top:10px}
      th,td{padding:8px;border:1px solid #ddd;text-align:left}
      .right{text-align:right}
      .total{font-weight:700;background:#f5f5f5}
    </style></head><body>';
    $html .= '<div class="header">';
    if ($logo_src) $html .= '<div class="logo"><img src="'.$logo_src.'" style="width:80px"></div>';
    $html .= '<div><div class="title">Resit Rasmi Monsoon Fun Run 2025</div><div class="muted">PASTI - Kuala Nerus</div></div></div>';

    $html .= '<hr>';
    $html .= '<p><strong>Order ID:</strong> ' . htmlspecialchars($order['order_id']) . '<br>';
    $html .= '<strong>Tarikh:</strong> ' . htmlspecialchars($order['payment_date'] ?? date('Y-m-d H:i:s')) . '</p>';

    $html .= '<h3>Butiran Pelanggan</h3>';
    $html .= '<p><strong>Nama:</strong> ' . htmlspecialchars($order['buyer_name']) . '<br>';
    $html .= '<strong>No. Telefon:</strong> ' . htmlspecialchars($order['buyer_phone']) . '</p>';

    $html .= '<h3>Butiran Pesanan</h3>';
    $html .= '<table><thead><tr><th>Item</th><th>Detail</th><th class="right">Harga (RM)</th></tr></thead><tbody>';

    $kategori = $meta['kategori'] ?? 'N/A';
    $size = $meta['size'] ?? 'N/A';
    $quantity = $meta['quantity'] ?? 1;
    $unit = isset($meta['unit_price']) ? number_format($meta['unit_price'],2) : '';
    $addon = (!empty($meta['addon_lengan']) && $meta['addon_lengan']) ? 'Ya' : 'Tidak';
    $addon_price = $addon==='Ya' ? 5 * $quantity : 0;

    $html .= '<tr><td>Baju Monsoon Fun Run</td><td>' . htmlspecialchars(strtoupper($kategori)) . ' - Saiz ' . htmlspecialchars($size) . ' - Qty: ' . intval($quantity) . ($addon==='Ya' ? ' (Lengan Panjang)' : '') . '</td><td class="right">' . number_format($order['amount'],2) . '</td></tr>';

    $html .= '</tbody><tfoot>';
    $html .= '<tr><td colspan="2" class="right">Jumlah</td><td class="right total">RM' . number_format($order['amount'],2) . '</td></tr>';
    $html .= '</tfoot></table>';

    $html .= '<p class="muted">Status: <strong>Bayaran Disahkan (ToyyibPay)</strong></p>';
    $html .= '<p>Resit ini dijana secara automatik oleh Sistem PASTI Kuala Nerus.</p>';

    $html .= '<div style="margin-top:30px">Disahkan oleh: <br><strong>PASTI Kuala Nerus</strong></div>';

    $html .= '</body></html>';

    $dompdf = new \Dompdf\Dompdf(['isRemoteEnabled' => true]);
    $dompdf->loadHtml($html);
    $dompdf->setPaper('A4','portrait');
    $dompdf->render();

    $outdir = __DIR__ . '/../receipts';
    if (!is_dir($outdir)) @mkdir($outdir, 0755, true);
    $filename = $order['order_id'] . '.pdf';
    $full = $outdir . '/' . $filename;
    file_put_contents($full, $dompdf->output());

    // return URL path assumption - adjust BASE_URL in config/app.php
    $base = defined('BASE_URL') ? BASE_URL : ((isset($_SERVER['REQUEST_SCHEME'])?$_SERVER['REQUEST_SCHEME']:'https') . '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost'));
    $url = rtrim($base, '/') . '/receipts/' . $filename;
    return ['ok'=>true,'path'=>$full,'url'=>$url];
}
